/********************************************************************************//**
 *
 * @file		LCD_16x2.c
 *
 ************************************************************************************
 *
 * @brief		
 *
 * @details		
 *
 * @note		
 *
 * @date		Aug 8, 2013
 * @author		dujalan
 * @version		
 * @copyright	Shalaka Technologies Pvt. Ltd.
 *
 ************************************************************************************
 ***********************************************************************************/




/*===================================================================================
									Include Files
===================================================================================*/
#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"

#include "driverlib/debug.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/interrupt.h"
#include "driverlib/gpio.h"

#include "LCD_16x2.h"


/*===================================================================================
								Function Definitions
===================================================================================*/

/*!
 ************************************************************************************
 * @name	LCD_16x2 API Functions
 * @{
 ************************************************************************************
 */

/*!
 ************************************************************************************
 * @brief		To generate a delay in Seconds.
 *
 * @param[in]	time		The number of seconds the delay is required.
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void delayS(unsigned int time)
{
	volatile unsigned long dS_t = 0;
	for(dS_t=0; dS_t<time*100000; dS_t++);
}

/*!
 ************************************************************************************
 *
 * @brief		This function is used internally to send an Enable High to Low
 * 				transition signal.
 *
 * \par Parameters
 * 				NONE
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void LCDSendEnable(void)
{
	GPIOPinWrite(LCD_CONTROL_PORT, EN, EN);
	delayS(1);
	GPIOPinWrite(LCD_CONTROL_PORT, EN, 0);
	delayS(1);
}

/*!
 ************************************************************************************
 *
 * @brief		This function is used to initialize the LCD with the default
 * 				settings.
 *
 * @details		It initializes the LCD with the following parameters:
 * 				-	4bit/8bit Mode (as per the #defined macro in the h file)
 * 				-	2 lines
 * 				-	5x8dots
 * 				-	Display On
 * 				-	Cursor On
 * 				-	Blink On
 *				-	Auto Cursor Increment
 *				-	Clear Display
 *				-	Return Cursor & Display to Home
 *
 * \par Parameters
 * 				NONE
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void LCDInit(void)
{
	SysCtlPeripheralEnable(LCD_CONTROL_PORT_ENABLE);
	SysCtlPeripheralEnable(LCD_DATA_PORT_ENABLE);

	GPIOPinTypeGPIOOutput(LCD_CONTROL_PORT, RS | EN);
	GPIOPinTypeGPIOOutput(LCD_DATA_PORT, 0xFF);

	#ifdef LCD_4BIT
		LCDSendCommand(LCD_FUNCTIONSET | LCD_4BITMODE | LCD_2LINE | LCD_5x8DOTS);
	#elif defined(LCD_8BIT)
		LCDSendCommand(LCD_FUNCTIONSET | LCD_8BITMODE | LCD_2LINE | LCD_5x8DOTS);
	#endif
	LCDSendCommand(LCD_DISPLAYCONTROL | LCD_DISPLAYON | LCD_CURSORON | LCD_BLINKON);
	LCDSendCommand(LCD_ENTRYMODESET | LCD_ENTRYRIGHT);
	LCDSendCommand(LCD_CLEARDISPLAY);
	LCDSendCommand(LCD_RETURNHOME);
}

/*!
 ************************************************************************************
 *
 * @brief		This function is used to send commands to the LCD.
 *
 * @details		It makes RS 0 for going into command mode and sends the inCommand
 * 				to the LCD.
 *
 * \param[in]	inCommand	The command according to the LCD Command Table which needs
 * 							to be sent to the LCD.
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void LCDSendCommand(char inCommand)
{
	#ifdef LCD_4BIT
		GPIOPinWrite(LCD_DATA_PORT, D4 | D5 | D6 | D7, inCommand & 0xF0);
		GPIOPinWrite(LCD_DATA_PORT, RS, 0);
		LCDSendEnable();

		GPIOPinWrite(LCD_DATA_PORT, D4 | D5 | D6 | D7, (inCommand & 0x0F) << 4);
		GPIOPinWrite(LCD_DATA_PORT, RS, 0);
		LCDSendEnable();
	#elif defined(LCD_8BIT)
		GPIOPinWrite(LCD_DATA_PORT, D0 | D1 | D2 | D3 | D4 | D5 | D6 | D7, inCommand);
		GPIOPinWrite(LCD_DATA_PORT, RS, 0);
		LCDSendEnable();
	#endif
}


/*!
 ************************************************************************************
 *
 * @brief		This function is used to send a single character data to the LCD.
 *
 * @details		It makes RS 1 for going into data mode and sends the inChar
 * 				to the LCD.
 *
 * \param[in]	inChar	The data which needs to be sent to the LCD.
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void LCDSendChar(char inChar)
{
	#ifdef LCD_4BIT
		GPIOPinWrite(LCD_DATA_PORT, D4 | D5 | D6 | D7, inChar & 0xF0);
		GPIOPinWrite(LCD_DATA_PORT, RS, RS);
		LCDSendEnable();

		GPIOPinWrite(LCD_DATA_PORT, D4 | D5 | D6 | D7, (inChar & 0x0F) << 4);
		GPIOPinWrite(LCD_DATA_PORT, RS, RS);
		LCDSendEnable();
	#elif defined(LCD_8BIT)
		GPIOPinWrite(LCD_DATA_PORT, D0 | D1 | D2 | D3 | D4 | D5 | D6 | D7, inChar);
		GPIOPinWrite(LCD_DATA_PORT, RS, RS);
		LCDSendEnable();
	#endif
}

/*!
 ************************************************************************************
 *
 * @brief		This function is used to send a string data to the LCD.
 *
 * @details		It makes RS 1 for going into data mode and sends the inString
 * 				to the LCD.
 *
 * \param[in]	inString	The data which needs to be sent to the LCD.
 *
 * @return		NONE
 *
 ************************************************************************************
 */
void LCDSendString(char *inString, char inRow, char inColumn)
{
	unsigned int LSS_inLen = strlen(inString);
	unsigned int LSS_i = 0;

	unsigned char LSS_Address = 0;

	switch(inRow)
	{
		case 0:
			LSS_Address = 0x00 + inColumn;
			break;
		case 1:
			LSS_Address = 0x40 + inColumn;
			break;
		default:
			LSS_Address = 0x00 + inColumn;
			break;
	}
	LCDSendCommand(LSS_Address);

	for(LSS_i = 0; LSS_i < LSS_inLen; LSS_i++)
	{
		LCDSendChar(inString[LSS_i]);
	}
}

/*!
 ************************************************************************************
 * @}
 ************************************************************************************
 */		// End of LCD_16x2 API Functions
